package de.davidartmann.rosa.db;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteException;
import android.util.Log;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import de.davidartmann.rosa.model.Person;

/**
 * DAO class for the {@link de.davidartmann.rosa.model.Person}.
 * Created by david on 02.03.16.
 */
public class PersonDao {

    private static final String TAG = PersonDao.class.getSimpleName();

    private SQLiteDatabase db;
    private RosaDbHelper dbHelper;
    private Context context;

    public PersonDao(Context context) {
        this.context = context;
        dbHelper = new RosaDbHelper(context);
    }

    public void openReadable() {
        try {
            db = dbHelper.getReadableDatabase();
        } catch (SQLiteException e) {
            Log.e(TAG, "Could not open DB for reading");
        }
    }

    public void openWritable() {
        try {
            db = dbHelper.getWritableDatabase();
        } catch (SQLiteException e) {
            Log.e(TAG, "Could not open DB for writing");
        }
    }

    public void close() {
        if (dbHelper != null) {
            dbHelper.close();
        } else {
            Log.w(TAG, "dbHelper was null, while trying to close it");
        }
    }

    /**
     * Creates a new Person in the db.
     * @param person given model object.
     * @return inserted object.
     */
    public Person create(Person person) {
        ContentValues contentValues = new ContentValues();
        contentValues.put(Person.COLUMN_CREATETIME, new Date().getTime());
        contentValues.put(Person.COLUMN_NAME, person.getName());
        contentValues.put(Person.COLUMN_ADDRESS, person.getAddress());
        contentValues.put(Person.COLUMN_PHONE, person.getPhone());
        contentValues.put(Person.COLUMN_EMAIL, person.getEmail());
        contentValues.put(Person.COLUMN_PRICE, person.getPrice());
        contentValues.put(Person.COLUMN_MISC, person.getMisc());
        contentValues.put(Person.COLUMN_CATEGORY, person.getCategory());
        contentValues.put(Person.COLUMN_ACTIVE, person.isActive());
        contentValues.put(Person.COLUMN_PICTUREURL, person.getPictureUrl());
        contentValues.put(Person.COLUMN_POSITION, person.getPosition());
        contentValues.put(Person.COLUMN_TABTEXT, person.getTabText());
        long insertId = db.insert(RosaDbHelper.TABLE_PERSON, null, contentValues);
        Cursor cursor = db.query(RosaDbHelper.TABLE_PERSON, Person.allColumns,
                Person.COLUMN_ID + "=" + insertId, null, null, null, null);
        cursor.moveToFirst();
        person = cursorToPerson(cursor);
        cursor.close();
        return person;
    }

    /**
     * @return all Persons from the db.
     */
    public List<Person> findAll() {
        List<Person> persons = new ArrayList<>();
        Cursor cursor = db.query(RosaDbHelper.TABLE_PERSON, Person.allColumns, null, null, null,
                null, null);
        cursor.moveToFirst();
        while (!cursor.isAfterLast()) {
            Person person = cursorToPerson(cursor);
            persons.add(person);
            cursor.moveToNext();
        }
        cursor.close();
        return persons;
    }

    /**
     * @return all active Persons from the db.
     */
    public List<Person> findAllActive() {
        List<Person> persons = new ArrayList<>();
        Cursor cursor = db.query(RosaDbHelper.TABLE_PERSON, Person.allColumns,
                Person.COLUMN_ACTIVE + "=1", null, null, null, Person.COLUMN_POSITION);
        cursor.moveToFirst();
        while (!cursor.isAfterLast()) {
            Person person = cursorToPerson(cursor);
            persons.add(person);
            cursor.moveToNext();
        }
        cursor.close();
        return persons;
    }

    /**
     * @return all inactive Persons from the db.
     */
    public List<Person> findAllInactive() {
        List<Person> persons = new ArrayList<>();
        Cursor cursor = db.query(RosaDbHelper.TABLE_PERSON, Person.allColumns,
                Person.COLUMN_ACTIVE + "=0", null, null, null, null);
        cursor.moveToFirst();
        while (!cursor.isAfterLast()) {
            Person person = cursorToPerson(cursor);
            persons.add(person);
            cursor.moveToNext();
        }
        cursor.close();
        return persons;
    }

    /**
     * Method to find a specific Person from the db by its id.
     * @param id given id of Person.
     * @return searched Person or null.
     */
    private Person findOne(int id) {
        Person person = null;
        Cursor cursor = db.query(RosaDbHelper.TABLE_PERSON, Person.allColumns,
                Person.COLUMN_ID + "=" + id, null, null, null, null);
        if (cursor != null) {
            boolean moveSuccess = cursor.moveToFirst();
            if (moveSuccess) {
                person = cursorToPerson(cursor);
                cursor.close();
            } else {
                Log.w(TAG, "cursor move to first not succeeded");
            }
        } else {
            Log.e(TAG, "cursor is null");
        }
        return person;
    }

    /**
     * Method to update a Person.
     * @param person given Person with new information.
     * @return updated Person.
     */
    public Person update(Person person) {
        ContentValues contentValues = new ContentValues();
        contentValues.put(Person.COLUMN_CREATETIME, person.getCreateTime());
        contentValues.put(Person.COLUMN_LASTUPDATE, new Date().getTime());
        contentValues.put(Person.COLUMN_NAME, person.getName());
        contentValues.put(Person.COLUMN_ADDRESS, person.getAddress());
        contentValues.put(Person.COLUMN_PHONE, person.getPhone());
        contentValues.put(Person.COLUMN_EMAIL, person.getEmail());
        contentValues.put(Person.COLUMN_PRICE, person.getPrice());
        contentValues.put(Person.COLUMN_MISC, person.getMisc());
        contentValues.put(Person.COLUMN_CATEGORY, person.getCategory());
        contentValues.put(Person.COLUMN_ACTIVE, person.isActive());
        contentValues.put(Person.COLUMN_PICTUREURL, person.getPictureUrl());
        contentValues.put(Person.COLUMN_POSITION, person.getPosition());
        contentValues.put(Person.COLUMN_TABTEXT, person.getTabText());
        int numberOfRows = db.update(RosaDbHelper.TABLE_PERSON, contentValues,
                Person.COLUMN_ID + "=" + person.get_id(), null);
        Person retPerson = findOne(person.get_id());
        if (numberOfRows > 1) {
            Log.w(TAG, "more than one person updated");
        } else if (numberOfRows < 1) {
            Log.w(TAG, "no person updated");
        } else {
            Log.d(TAG, "updated Person with id "+retPerson.get_id());
        }

        return retPerson;
    }

    /**
     * Method to delete a Person.
     * @param person given person to delete.
     * @return true if successfully deleted (also if more than one Person), false otherwise.
     */
    public boolean deletePerson(Person person) {
        int numberOfRowsAffected = db.delete(RosaDbHelper.TABLE_PERSON,
                Person.COLUMN_ID + "=" + person.get_id(), null);
        if (numberOfRowsAffected < 0) {
            Log.w(TAG, "no person deleted");
            return false;
        } else if (numberOfRowsAffected > 1) {
            Log.w(TAG, "more than one person deleted");
            return true;
        }
        Log.d(TAG, "deleted person with id "+person.get_id());
        return true;
    }

    /**
     * Method to soft delete a person.
     * This will set its status to false (sqlite stores a zero).
     * @param person given person.
     * @return soft deleted person.
     */
    public Person softDelete(Person person) {
        person.setActive(false);
        person.setPosition(0);
        return update(person);
    }

    /**
     * Method to permanently delete all inactive Persons from the db.
     * @return number of deleted Persons.
     */
    public int deleteAllInactive() {
        return db.delete(RosaDbHelper.TABLE_PERSON, Person.COLUMN_ACTIVE + "=0", null);
    }

    /**
     * Helper method to convert a cursor object into a person.
     * @param cursor given cursor.
     * @return person object or null.
     */
    private Person cursorToPerson(Cursor cursor) {
        Person person = new Person();
        person.set_id(cursor.getInt(0));
        //Log.d(TAG, "cursorToPerson with id value:"+ person.get_id());
        person.setCreateTime(cursor.getLong(1));
        person.setUpdateTime(cursor.getLong(2));
        person.setName(cursor.getString(3));
        person.setAddress(cursor.getString(4));
        person.setPhone(cursor.getString(5));
        person.setEmail(cursor.getString(6));
        person.setPrice(cursor.getString(7));
        person.setMisc(cursor.getString(8));
        person.setCategory(cursor.getInt(9));
        person.setActive(cursor.getInt(10) == 1);
        person.setPictureUrl(cursor.getString(11));
        person.setPosition(cursor.getInt(12));
        person.setTabText(cursor.getString(13));
        return person;
    }
}
